<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display a listing of users for the current Travel Agent (Company).
     * (Requirement 1: Admin can list users)
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // Scope query to only retrieve users belonging to the Admin's travel_agent_id
        $users = User::where('travel_agent_id', $request->user()->travel_agent_id)
                     ->orderBy('is_admin', 'desc') // Admins first
                     ->orderBy('name')
                     ->get(['id', 'name', 'email', 'mobile', 'is_admin', 'is_active', 'last_login_at']);

        return response()->json($users);
    }

    /**
     * Store a newly created user (Requirement 1: Admin can Add user).
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $admin = $request->user();

        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email',
            'mobile' => 'required|string|max:255|unique:users,mobile',
            'is_admin' => 'sometimes|boolean',
            'password' => 'required|string|min:8',
        ]);

        $user = User::create([
            'travel_agent_id' => $admin->travel_agent_id, // Assign to Admin's company
            'name' => $validatedData['name'],
            'email' => $validatedData['email'],
            'mobile' => $validatedData['mobile'],
            'password' => Hash::make($validatedData['password']),
            'is_admin' => $validatedData['is_admin'] ?? false,
            'is_active' => true, 
        ]);

        return response()->json([
            'message' => 'User created successfully.',
            'user' => $user->only(['id', 'name', 'email', 'mobile', 'is_admin', 'is_active'])
        ], 201);
    }

    /**
     * Display the specified user.
     * @param \App\Models\User $user
     * @return \Illuminate\Http\JsonResponse
     */
    public function show(User $user)
    {
        // SECURITY: Ensure the user belongs to the Admin's company
        if ($user->travel_agent_id !== auth()->user()->travel_agent_id) {
            abort(403, 'Unauthorized action: User does not belong to your company.');
        }

        return response()->json($user->only(['id', 'name', 'email', 'mobile', 'is_admin', 'is_active', 'last_login_at', 'created_at']));
    }

    /**
     * Update the specified user (Requirement 1: Admin can Edit user).
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, User $user)
    {
        // SECURITY: Ensure the user belongs to the Admin's company
        if ($user->travel_agent_id !== $request->user()->travel_agent_id) {
            abort(403, 'Unauthorized action: User does not belong to your company.');
        }

        // SECURITY: Prevent an admin from deactivating/demoting themselves
        if ($user->id === $request->user()->id) {
            if ($request->has('is_admin') && $request->is_admin === false) {
                 return response()->json(['message' => 'Cannot remove your own admin rights.'], 403);
            }
            if ($request->has('is_active') && $request->is_active === false) {
                 return response()->json(['message' => 'Cannot deactivate your own account.'], 403);
            }
        }

        $validatedData = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => ['sometimes', 'string', 'email', 'max:255', Rule::unique('users', 'email')->ignore($user->id)],
            'mobile' => ['sometimes', 'string', 'max:255', Rule::unique('users', 'mobile')->ignore($user->id)],
            'is_admin' => 'sometimes|boolean',
            'is_active' => 'sometimes|boolean',
            'password' => 'nullable|string|min:8',
        ]);

        // Handle password update
        if (isset($validatedData['password'])) {
            $validatedData['password'] = Hash::make($validatedData['password']);
        } else {
            unset($validatedData['password']);
        }
        
        // Requirement 2: If user is deactivated, immediately log them out.
        if (isset($validatedData['is_active']) && $validatedData['is_active'] === false) {
            $user->tokens()->delete();
        }

        $user->update($validatedData);

        return response()->json([
            'message' => 'User updated successfully.',
            'user' => $user->only(['id', 'name', 'email', 'mobile', 'is_admin', 'is_active'])
        ]);
    }

    /**
     * Delete the specified user (Requirement 1: Admin can Delete user).
     * @param \App\Models\User $user
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(User $user)
    {
        // SECURITY: Ensure the user belongs to the Admin's company
        if ($user->travel_agent_id !== auth()->user()->travel_agent_id) {
            abort(403, 'Unauthorized action: User does not belong to your company.');
        }

        // SECURITY: Prevent admin from deleting themselves
        if ($user->id === auth()->user()->id) {
            return response()->json(['message' => 'Cannot delete your own account.'], 403);
        }

        // Remote logout before deletion
        $user->tokens()->delete();
        $user->delete();

        return response()->json(['message' => 'User deleted successfully.']);
    }
    
    /**
     * Toggles the user's active status and performs an immediate remote logout.
     * (Requirement 2 & 3: Admin can deactivate/activate and immediately log out)
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggleStatus(Request $request, User $user)
    {
        // SECURITY: Policy Check
        if ($user->travel_agent_id !== $request->user()->travel_agent_id) {
            abort(403, 'Unauthorized action: User does not belong to your company.');
        }
        
        // SECURITY: Prevent admin from changing their own status via this action
        if ($user->id === $request->user()->id) {
            return response()->json(['message' => 'Cannot change status of your own account via this action.'], 403);
        }
        
        $user->is_active = !$user->is_active;
        $user->save();
        
        $action = $user->is_active ? 'Reactivated' : 'Deactivated';

        // Requirement 2 & 3: If deactivated, perform remote logout
        if (!$user->is_active) {
            $user->tokens()->delete();
        }

        return response()->json([
            'message' => "User successfully {$action} and logged out from all devices.",
            'user' => $user->only(['id', 'name', 'is_active'])
        ]);
    }
}
