<?php

namespace App\Http\Controllers;

use App\Models\TravelAgent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class TravelAgentController extends Controller
{
    public function index()
    {
        // Return a list of all travel agents (optional)
        return TravelAgent::all();
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // 1. Validate the incoming data
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:travel_agents,email',
            'phone' => 'nullable|string',
            'contact_person_name' => 'nullable|string|max:255',
            'gst_number' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'logo_file' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'rules_and_regulations' => 'nullable|string',
            'terms_and_conditions' => 'nullable|string',
            'cancellation_policy' => 'nullable|string',
            'inclusions' => 'nullable|json',
            'exclusions' => 'nullable|json',
            'mail_mailer' => 'nullable|string',
            // ... all other fields
        ]);
        
        // 2. Handle the logo file upload
        if ($request->hasFile('logo_file')) {
            $path = $request->file('logo_file')->store('logos', 'public');
            $validatedData['logo_file'] = $path;
        }

        // 3. Create the new TravelAgent record
        $travelAgent = TravelAgent::create($validatedData);

        return response()->json($travelAgent, 201);
    }

    public function show(string $id)
    {
        // The CompanyController already handles this, but you could implement it here too
        $travelAgent = TravelAgent::find($id);

        if (!$travelAgent) {
            return response()->json(['message' => 'Travel Agent not found'], 404);
        }

        return response()->json($travelAgent, 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $travelAgent = TravelAgent::find($id);
        if (!$travelAgent) {
            return response()->json(['message' => 'Travel Agent not found'], 404);
        }

        // 1. Validate the incoming data for the update
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:travel_agents,email,' . $travelAgent->id,
            'phone' => 'nullable|string',
            'contact_person_name' => 'nullable|string|max:255',
            'gst_number' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'logo_file' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'rules_and_regulations' => 'nullable|string',
            'terms_and_conditions' => 'nullable|string',
            'cancellation_policy' => 'nullable|string',
            'inclusions' => 'nullable|json',
            'exclusions' => 'nullable|json',
            'mail_mailer' => 'nullable|string',
            // ... all other fields
        ]);
        
        // 2. Handle new logo file upload
        if ($request->hasFile('logo_file')) {
            // Delete old file if it exists
            if ($travelAgent->logo_file) {
                Storage::disk('public')->delete($travelAgent->logo_file);
            }
            $path = $request->file('logo_file')->store('logos', 'public');
            $validatedData['logo_file'] = $path;
        }

        // 3. Update the TravelAgent record
        $travelAgent->update($validatedData);

        return response()->json($travelAgent, 200);
    }

    public function destroy(string $id)
    {
        $travelAgent = TravelAgent::find($id);
        if (!$travelAgent) {
            return response()->json(['message' => 'Travel Agent not found'], 404);
        }

        $travelAgent->delete();

        return response()->json(['message' => 'Travel Agent deleted successfully'], 200);
    }
}