<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\TopUpOption;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class TopUpController extends Controller
{
    public function index()
    {
        $topUps = TopUpOption::orderBy('words')->get();
        return response()->json($topUps);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'words' => 'required|integer|min:100',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'is_best_value' => 'boolean',
        ]);

        $topUp = TopUpOption::create($validated);

        return response()->json([
            'message' => 'Top-up option created successfully',
            'data' => $topUp
        ], 201);
    }

    public function update(Request $request, $id)
    {
        $topUp = TopUpOption::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'words' => 'required|integer|min:100',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string|max:255',
            'is_active' => 'boolean',
            'is_best_value' => 'boolean',
        ]);

        $topUp->update($validated);

        return response()->json([
            'message' => 'Top-up option updated successfully',
            'data' => $topUp
        ]);
    }

    public function toggleStatus($id)
    {
        $topUp = TopUpOption::findOrFail($id);
        $topUp->update(['is_active' => !$topUp->is_active]);

        $action = $topUp->is_active ? 'activated' : 'deactivated';

        return response()->json([
            'message' => "Top-up option {$action} successfully",
            'data' => $topUp
        ]);
    }

    public function setBestValue($id)
    {
        // Remove best value from all options
        TopUpOption::where('is_best_value', true)->update(['is_best_value' => false]);
        
        // Set new best value
        $topUp = TopUpOption::findOrFail($id);
        $topUp->update(['is_best_value' => true]);

        return response()->json([
            'message' => 'Best value option updated successfully',
            'data' => $topUp
        ]);
    }
}