<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\AiPlan;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class SubscriptionPlanController extends Controller
{
    public function index()
    {
        $plans = AiPlan::orderBy('sort_order')->get();
        return response()->json($plans);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:ai_plans,name',
            'display_name' => 'required|string|max:255',
            'monthly_words' => 'required|integer|min:0',
            'carry_forward_months' => 'required|integer|min:0',
            'top_up_rate' => 'required|numeric|min:0',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'free_words_included' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'sort_order' => 'required|integer',
            'features' => 'nullable|array',
        ]);

        $plan = AiPlan::create($validated);

        return response()->json([
            'message' => 'Subscription plan created successfully',
            'data' => $plan
        ], 201);
    }

    public function update(Request $request, $id)
    {
        $plan = AiPlan::findOrFail($id);

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('ai_plans')->ignore($plan->id)],
            'display_name' => 'required|string|max:255',
            'monthly_words' => 'required|integer|min:0',
            'carry_forward_months' => 'required|integer|min:0',
            'top_up_rate' => 'required|numeric|min:0',
            'price' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'free_words_included' => 'required|integer|min:0',
            'is_active' => 'boolean',
            'sort_order' => 'required|integer',
            'features' => 'nullable|array',
        ]);

        $plan->update($validated);

        return response()->json([
            'message' => 'Subscription plan updated successfully',
            'data' => $plan
        ]);
    }

    public function toggleStatus($id)
    {
        $plan = AiPlan::findOrFail($id);
        $plan->update(['is_active' => !$plan->is_active]);

        $action = $plan->is_active ? 'activated' : 'deactivated';

        return response()->json([
            'message' => "Subscription plan {$action} successfully",
            'data' => $plan
        ]);
    }

    public function destroy($id)
    {
        $plan = AiPlan::findOrFail($id);
        
        // Check if plan has active subscriptions
        if ($plan->subscriptions()->where('status', 'active')->exists()) {
            return response()->json([
                'message' => 'Cannot delete plan with active subscriptions'
            ], 422);
        }

        $plan->delete();

        return response()->json([
            'message' => 'Subscription plan deleted successfully'
        ]);
    }
}