<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\TravelAgent;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class PaymentAnalyticsController extends Controller
{
    public function overview(Request $request)
    {
        $period = $request->get('period', 'monthly'); // daily, weekly, monthly, yearly
        
        $query = Payment::where('status', 'captured');

        // Date filtering
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('created_at', [
                Carbon::parse($request->start_date)->startOfDay(),
                Carbon::parse($request->end_date)->endOfDay()
            ]);
        } else {
            // Default to last 30 days
            $query->where('created_at', '>=', Carbon::now()->subDays(30));
        }

        // Travel Agent filter
        if ($request->has('travel_agent_id') && $request->travel_agent_id) {
            $query->where('travel_agent_id', $request->travel_agent_id);
        }

        $payments = $query->with('travelAgent')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Summary statistics
        $summary = [
            'total_revenue' => $query->sum('amount'),
            'total_payments' => $query->count(),
            'average_payment' => $query->avg('amount'),
        ];

        return response()->json([
            'payments' => $payments,
            'summary' => $summary,
        ]);
    }

    public function revenueChart(Request $request)
    {
        $period = $request->get('period', 'monthly');

        if ($period === 'monthly') {
            $data = Payment::where('status', 'captured')
                ->where('created_at', '>=', Carbon::now()->subMonths(12))
                ->selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, SUM(amount) as revenue, COUNT(*) as count')
                ->groupBy('year', 'month')
                ->orderBy('year', 'asc')
                ->orderBy('month', 'asc')
                ->get();
        } elseif ($period === 'weekly') {
            $data = Payment::where('status', 'captured')
                ->where('created_at', '>=', Carbon::now()->subWeeks(12))
                ->selectRaw('YEAR(created_at) as year, WEEK(created_at) as week, SUM(amount) as revenue, COUNT(*) as count')
                ->groupBy('year', 'week')
                ->orderBy('year', 'asc')
                ->orderBy('week', 'asc')
                ->get();
        } else { // daily
            $data = Payment::where('status', 'captured')
                ->where('created_at', '>=', Carbon::now()->subDays(30))
                ->selectRaw('DATE(created_at) as date, SUM(amount) as revenue, COUNT(*) as count')
                ->groupBy('date')
                ->orderBy('date', 'asc')
                ->get();
        }

        return response()->json($data);
    }

    public function topAgentsByRevenue()
    {
        $topAgents = Payment::where('status', 'captured')
            ->selectRaw('travel_agent_id, SUM(amount) as total_revenue, COUNT(*) as payment_count')
            ->with('travelAgent')
            ->groupBy('travel_agent_id')
            ->orderBy('total_revenue', 'desc')
            ->limit(10)
            ->get();

        return response()->json($topAgents);
    }
}