<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Service;

class ServiceController extends Controller
{
    /**
     * Display a listing of all services
     */
    public function index()
    {
        $services = Service::with('travelAgent')->get();
        return response()->json($services);
    }

    /**
     * Store a newly created service
     */
    public function store(Request $request)
    {
        $request->validate([
            'travel_agent_id' => 'required|exists:travel_agents,id',
            'name' => 'required|string|max:255',
            'type_variation' => 'required|array'
        ]);

        $service = Service::create([
            'travel_agent_id' => $request->travel_agent_id,
            'name' => $request->name,
            'type_variation' => $request->type_variation
        ]);

        return response()->json($service, 201);
    }

    /**
     * Display the specified service
     */
    public function show(string $id)
    {
        $service = Service::with('travelAgent')->findOrFail($id);
        return response()->json($service);
    }

    /**
     * Update the specified service
     */
    public function update(Request $request, string $id)
    {
        $service = Service::findOrFail($id);

        $request->validate([
            'travel_agent_id' => 'sometimes|exists:travel_agents,id',
            'name' => 'sometimes|string|max:255',
            'type_variation' => 'sometimes|array'
        ]);

        $service->update($request->only('travel_agent_id', 'name', 'type_variation'));

        return response()->json($service);
    }

    /**
     * Remove the specified service
     */
    public function destroy(string $id)
    {
        $service = Service::findOrFail($id);
        $service->delete();

        return response()->json(['message' => 'Service deleted successfully']);
    }

    /**
     * POS Search API for services
     */
    public function search(Request $request)
    {
        $request->validate([
            'query' => 'required|string'
        ]);

        $services = Service::where('name', 'like', '%' . $request->query . '%')
            ->orWhereJsonContains('type_variation->type', $request->query)
            ->get();

        return response()->json($services);
    }
}

