<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Razorpay\Api\Api;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\Payment;
use App\Models\AiPlan;
use App\Models\TopUpOption;
use App\Models\AiKey;
use App\Models\AiSubscription;
use Illuminate\Support\Str;
use Exception;

class RazorpayController extends Controller
{
    private $razorpay;

    public function __construct()
    {
        try {
            $keyId = env('RAZORPAY_KEY_ID');
            $keySecret = env('RAZORPAY_KEY_SECRET');
            
            if (empty($keyId) || empty($keySecret)) {
                throw new \Exception('Razorpay credentials are not configured in environment variables');
            }
            
            $this->razorpay = new Api($keyId, $keySecret);
            
            // Test the connection
            $this->razorpay->order->create([
                'receipt' => 'test_connection',
                'amount' => 100,
                'currency' => 'INR',
                'payment_capture' => 1
            ]);
            
        } catch (Exception $e) {
            Log::error('Razorpay SDK initialization failed: ' . $e->getMessage());
            throw new \Exception('Razorpay SDK initialization failed: ' . $e->getMessage());
        }
    }

    /**
     * Create a Razorpay Order
     */
    public function createOrder(Request $request)
    {
        Log::info('Razorpay createOrder called', $request->all());

        $validator = Validator::make($request->all(), [
            'travel_agent_id' => 'required|exists:travel_agents,id',
            'entity_type' => 'required|in:plan,top_up',
            'entity_id' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            Log::error('Validation failed', $validator->errors()->toArray());
            return response()->json(['error' => 'Validation failed', 'messages' => $validator->errors()], 422);
        }

        try {
            $entityType = $request->input('entity_type');
            $entityId = $request->input('entity_id');
            $agentId = $request->input('travel_agent_id');

            Log::info("Processing order for entity: $entityType, ID: $entityId, Agent: $agentId");

            $entity = null;
            $amountInRupees = 0;

            if ($entityType === 'plan') {
                $entity = AiPlan::find($entityId);
                if (!$entity) {
                    Log::error("Plan not found: $entityId");
                    return response()->json(['error' => 'Plan not found'], 404);
                }
                // For plans, use the fixed price
                $amountInRupees = $entity->price;
                
            } else {
                $entity = TopUpOption::find($entityId);
                if (!$entity) {
                    Log::error("Top-up option not found: $entityId");
                    return response()->json(['error' => 'Top-up option not found'], 404);
                }
                
                // ✅ FIXED: Calculate dynamic price for top-ups based on user's plan
                $aiKey = AiKey::where('travel_agent_id', $agentId)->first();
                if (!$aiKey) {
                    Log::error("AI Key not found for travel agent: $agentId");
                    return response()->json(['error' => 'AI configuration not found'], 404);
                }
                
                $topUpRate = $aiKey->getCurrentTopUpRate();
                $amountInRupees = $entity->words * $topUpRate;
                
                Log::info("Dynamic top-up pricing - Words: {$entity->words}, Rate: $topUpRate, Total: $amountInRupees");
            }

            // Validate amount
            if ($amountInRupees <= 0) {
                Log::error('Invalid amount: ' . $amountInRupees);
                return response()->json(['error' => 'Invalid amount'], 400);
            }
            
            $amountInPaise = $amountInRupees * 100;

            // Razorpay requires minimum 1 INR (100 paise)
            if ($amountInPaise < 100) {
                $amountInPaise = 100;
            }

            $orderData = [
                'receipt'         => 'order_' . $agentId . '_' . time(),
                'amount'          => $amountInPaise,
                'currency'        => 'INR',
                'payment_capture' => 1
            ];

            Log::info('Creating Razorpay order with data:', $orderData);

            $razorpayOrder = $this->razorpay->order->create($orderData);

            Log::info('Razorpay order created:', ['order_id' => $razorpayOrder['id']]);

            // Log the payment attempt in our database
            Payment::create([
                'travel_agent_id' => $agentId,
                'razorpay_order_id' => $razorpayOrder['id'],
                'amount' => $amountInRupees,
                'currency' => 'INR',
                'status' => 'created',
                'entity_type' => $entityType,
                'entity_id' => $entityId,
            ]);

            return response()->json([
                'order_id' => $razorpayOrder['id'],
                'amount' => $amountInPaise,
                'currency' => 'INR',
                'entity_name' => $entity->display_name ?? $entity->title,
                'razorpay_key' => env('RAZORPAY_KEY_ID'),
                // ✅ ADD: Return calculated price for frontend verification
                'calculated_price' => $amountInRupees,
            ]);

        } catch (Exception $e) {
            Log::error('Razorpay Order Creation Failed: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json(['error' => 'Could not create payment order. ' . $e->getMessage()], 500);
        }
    }

    /**
     * Verify Payment Signature
     */
    public function verifyPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'razorpay_order_id' => 'required|string',
            'razorpay_payment_id' => 'required|string',
            'razorpay_signature' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Invalid payment details', 'messages' => $validator->errors()], 422);
        }

        $payment = Payment::where('razorpay_order_id', $request->razorpay_order_id)->firstOrFail();

        try {
            $attributes = [
                'razorpay_order_id' => $request->razorpay_order_id,
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature
            ];
            
            $this->razorpay->utility->verifyPaymentSignature($attributes);

            $payment->update([
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature,
                'status' => 'captured',
            ]);

            $this->provisionService($payment);
            
            return response()->json(['status' => 'success', 'message' => 'Payment successful and service activated.']);

        } catch(Exception $e) {
            Log::error('Razorpay Signature Verification Failed: ' . $e->getMessage());
            $payment->update(['status' => 'failed']);
            return response()->json(['status' => 'error', 'message' => 'Payment verification failed.'], 400);
        }
    }

    /**
     * Provision the service after successful payment
     */
    private function provisionService(Payment $payment)
    {
        if ($payment->entity_type === 'plan') {
            $this->activateSubscription($payment->travel_agent_id, $payment->entity_id, $payment->razorpay_payment_id);
        } elseif ($payment->entity_type === 'top_up') {
            $this->addTopUpWords($payment->travel_agent_id, $payment->entity_id);
        }
    }

    /**
     * Activate a new subscription plan for a travel agent
     */
    private function activateSubscription($agentId, $planId, $paymentId)
    {
        $newPlan = AiPlan::findOrFail($planId);
        
        AiSubscription::where('travel_agent_id', $agentId)
            ->where('status', 'active')
            ->update(['status' => 'canceled', 'end_date' => now()]);
        
        AiSubscription::create([
            'travel_agent_id' => $agentId,
            'plan_id' => $planId,
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addMonth(),
            'billing_cycle_start' => now(),
            'billing_cycle_end' => now()->addMonth(),
            'words_remaining' => $newPlan->monthly_words,
            'words_used' => 0,
            'price' => $newPlan->price,
            'payment_status' => 'succeeded',
            'payment_id' => $paymentId,
        ]);

        Log::info("Subscription Activated for Agent ID: {$agentId}, Plan: {$newPlan->name}");
    }

    /**
     * Add top-up words to a travel agent's account
     */
    private function addTopUpWords($agentId, $topUpId)
    {
        $topUpOption = TopUpOption::findOrFail($topUpId);
        $aiKey = AiKey::where('travel_agent_id', $agentId)->firstOrFail();
        
        // Get the current top-up rate based on user's plan
        $topUpRate = $aiKey->getCurrentTopUpRate();
        
        // Calculate the actual price based on words and rate
        $calculatedPrice = $topUpOption->words * $topUpRate;
        
        // Update the top-up words
        $aiKey->increment('top_up_words', $topUpOption->words);

        Log::info("Top-Up Added for Agent ID: {$agentId}, Words: {$topUpOption->words}, Rate: ₹{$topUpRate}/word, Calculated Price: ₹{$calculatedPrice}");
    }
}