<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\InvoiceDay;
use App\Models\InvoiceDayService;
use App\Models\InvoiceDayHotel;
use App\Models\InvoiceDayExtraService;
use App\Models\DayDescription; // ADD THIS IMPORT

class InvoiceDayController extends Controller
{
    // List all days (optionally filtered by invoice_id)
    public function index(Request $request)
    {
        $query = InvoiceDay::with([
            'services.service', 
            'hotels.hotel', 
            'extraServices.extraService',
            'dayDescriptions' // ADD THIS
        ]);

        if ($request->has('invoice_id')) {
            $query->where('invoice_id', $request->invoice_id);
        }

        return response()->json($query->orderBy('day_number')->get());
    }

    // Create a new day for an invoice
    public function store(Request $request)
    {
        $request->validate([
            'invoice_id' => 'required|exists:invoices,id',
            'day_date'   => 'required|date'
        ]);

        // Auto-increment day number for this invoice
        $dayNumber = InvoiceDay::where('invoice_id', $request->invoice_id)->count() + 1;

        $day = InvoiceDay::create([
            'invoice_id' => $request->invoice_id,
            'day_number' => $dayNumber,
            'day_date'   => $request->day_date,
        ]);

        return response()->json($day, 201);
    }

    // Show details of a day
    public function show($id)
    {
        $day = InvoiceDay::with([
            'services.service', 
            'hotels.hotel', 
            'extraServices.extraService',
            'dayDescriptions' // ADD THIS
        ])->findOrFail($id);
        
        return response()->json($day);
    }

    // Update day_date
    public function update(Request $request, $id)
    {
        $request->validate([
            'day_date' => 'required|date',
        ]);

        $day = InvoiceDay::findOrFail($id);
        $day->update(['day_date' => $request->day_date]);

        return response()->json($day);
    }

    // Delete a day along with its nested data
    public function destroy($id)
    {
        $day = InvoiceDay::findOrFail($id);
        $day->services()->delete();
        $day->hotels()->delete();
        $day->extraServices()->delete();
        $day->dayDescriptions()->detach(); // ADD THIS
        $day->delete();

        return response()->json(['message' => 'Invoice day deleted successfully']);
    }

    // Add a service to a day
    public function addService(Request $request, $day_id)
    {
        $request->validate([
            'service_id' => 'required|exists:services,id',
            'variation' => 'required|string',
            'qty' => 'required|integer|min:1',
            'unit_price' => 'required|numeric|min:0'
        ]);

        $service = InvoiceDayService::create([
            'invoice_day_id' => $day_id,
            'service_id' => $request->service_id,
            'variation' => $request->variation,
            'qty' => $request->qty,
            'unit_price' => $request->unit_price
        ]);

        return response()->json($service, 201);
    }

    // Add a hotel to a day
    public function addHotel(Request $request, $day_id)
    {
        $request->validate([
            'hotel_id' => 'required|exists:hotels,id',
            'room_type' => 'required|string',
            'extra_bed' => 'nullable|boolean',
            'price' => 'required|numeric|min:0',
            'qty' => 'required|integer|min:1'
        ]);

        $hotel = InvoiceDayHotel::create([
            'invoice_day_id' => $day_id,
            'hotel_id' => $request->hotel_id,
            'room_type' => $request->room_type,
            'extra_bed' => $request->extra_bed ?? 0,
            'price' => $request->price,
            'qty' => $request->qty
        ]);

        return response()->json($hotel, 201);
    }

    // Add an extra service to a day
    public function addExtraService(Request $request, $day_id)
    {
        $request->validate([
            'extra_service_id' => 'required|exists:extra_services,id',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1'
        ]);

        $extra = InvoiceDayExtraService::create([
            'invoice_day_id' => $day_id,
            'extra_service_id' => $request->extra_service_id,
            'description' => $request->description,
            'price' => $request->price,
            'quantity' => $request->quantity ?? 1
        ]);

        return response()->json($extra, 201);
    }

    // NEW METHOD: Add day descriptions to a day
    public function addDayDescription(Request $request, $day_id)
    {
        $request->validate([
            'day_description_id' => 'required|exists:day_descriptions,id'
        ]);

        $day = InvoiceDay::findOrFail($day_id);
        $day->dayDescriptions()->attach($request->day_description_id);

        return response()->json(['message' => 'Day description added successfully'], 201);
    }

    // NEW METHOD: Remove day description from a day
    public function removeDayDescription(Request $request, $day_id)
    {
        $request->validate([
            'day_description_id' => 'required|exists:day_descriptions,id'
        ]);

        $day = InvoiceDay::findOrFail($day_id);
        $day->dayDescriptions()->detach($request->day_description_id);

        return response()->json(['message' => 'Day description removed successfully'], 200);
    }

    // Get services of a specific day
    public function getServices($day_id)
    {
        $services = InvoiceDayService::where('invoice_day_id', $day_id)->with('service')->get();
        return response()->json($services);
    }

    // Get hotels of a specific day
    public function getHotels($day_id)
    {
        $hotels = InvoiceDayHotel::where('invoice_day_id', $day_id)->with('hotel')->get();
        return response()->json($hotels);
    }

    // Get extra services of a specific day
    public function getExtraServices($day_id)
    {
        $extras = InvoiceDayExtraService::where('invoice_day_id', $day_id)->with('extraService')->get();
        return response()->json($extras);
    }

    // NEW METHOD: Get day descriptions of a specific day
    public function getDayDescriptions($day_id)
    {
        $day = InvoiceDay::with('dayDescriptions')->findOrFail($day_id);
        return response()->json($day->dayDescriptions);
    }
}