<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Hotel;

class HotelController extends Controller
{
    /**
     * Display a listing of all hotels
     */
    public function index()
    {
        $hotels = Hotel::with('travelAgent')->get();
        return response()->json($hotels);
    }

    /**
     * Store a newly created hotel
     */
    public function store(Request $request)
    {
        $request->validate([
            'travel_agent_id' => 'required|exists:travel_agents,id',
            'name' => 'required|string|max:255',
            'room_variations' => 'required|array',
            'extra_bed_price' => 'required|numeric|min:0'
        ]);

        $hotel = Hotel::create([
            'travel_agent_id' => $request->travel_agent_id,
            'name' => $request->name,
            'room_variations' => $request->room_variations,
            'extra_bed_price' => $request->extra_bed_price
        ]);

        return response()->json($hotel, 201);
    }

    /**
     * Display the specified hotel
     */
    public function show(string $id)
    {
        $hotel = Hotel::with('travelAgent')->findOrFail($id);
        return response()->json($hotel);
    }

    /**
     * Update the specified hotel
     */
    public function update(Request $request, string $id)
    {
        $hotel = Hotel::findOrFail($id);

        $request->validate([
            'travel_agent_id' => 'sometimes|exists:travel_agents,id',
            'name' => 'sometimes|string|max:255',
            'room_variations' => 'sometimes|array',
            'extra_bed_price' => 'sometimes|numeric|min:0'
        ]);

        $hotel->update($request->only('travel_agent_id', 'name', 'room_variations', 'extra_bed_price'));

        return response()->json($hotel);
    }

    /**
     * Remove the specified hotel
     */
    public function destroy(string $id)
    {
        $hotel = Hotel::findOrFail($id);
        $hotel->delete();

        return response()->json(['message' => 'Hotel deleted successfully']);
    }

    /**
     * POS Search API for hotels
     */
    public function search(Request $request)
    {
        $request->validate([
            'query' => 'required|string'
        ]);

        $hotels = Hotel::where('name', 'like', '%' . $request->query . '%')
            ->orWhereJsonContains('room_variations->type', $request->query)
            ->get();

        return response()->json($hotels);
    }
}
