<?php
// app/Http/Controllers/AiGenerationController.php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use App\Models\AiKey;
use App\Models\AiUsageLog;
use App\Models\AiSubscription;
use App\Models\AiPlan;
use App\Models\Service;
use App\Models\Hotel;
use App\Models\ExtraService;
use App\Models\DayDescription;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Crypt;

class AiGenerationController extends Controller
{
    // Minimum word requirements for different page types
    const MIN_WORDS_DESCRIPTION = 75;
    const MIN_WORDS_ITINERARY = 250;

    public function generateDescription(Request $request)
    {
        return $this->handleGeneration($request, self::MIN_WORDS_DESCRIPTION, 'description');
    }

    public function generateItinerary(Request $request)
    {
        return $this->handleGeneration($request, self::MIN_WORDS_ITINERARY, 'itinerary');
    }

    private function handleGeneration(Request $request, $minWordsRequired, $generationType)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'places' => 'nullable|string|max:1000',
            'travel_agent_id' => 'required|exists:travel_agents,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => 'Validation failed',
                'messages' => $validator->errors()
            ], 422);
        }

        // Get travel_agent_id from request
        $travelAgentId = $request->travel_agent_id;
        $title = $request->title;
        $places = $request->places;

        // Get the AI key for this travel agent
        $aiKey = AiKey::where('travel_agent_id', $travelAgentId)->first();

        if (!$aiKey) {
            return response()->json([
                'error' => 'AI key not found for this travel agent'
            ], 404);
        }

        if (!$aiKey->is_active) {
            return response()->json([
                'error' => 'AI service is not active for your account'
            ], 403);
        }

        // Check if API key is set and valid
        if (empty($aiKey->gemini_api_key) || !$aiKey->hasValidApiKey()) {
            Log::error('Invalid or missing API key', [
                'travel_agent_id' => $travelAgentId,
                'has_valid_key' => $aiKey->hasValidApiKey()
            ]);
            
            return response()->json([
                'error' => 'API key not configured or invalid. Please contact administrator.'
            ], 500);
        }

        // Check if user has minimum words required for this generation type
        $totalAvailable = $aiKey->getTotalAvailableWords();
        if ($totalAvailable < $minWordsRequired) {
            $wordsNeeded = $minWordsRequired - $totalAvailable;
            $pageType = $this->getPageTypeName($minWordsRequired);
            
            return response()->json([
                'error' => "Insufficient credits. You need {$wordsNeeded} more words to generate {$pageType}.",
                'required_words' => $minWordsRequired,
                'available_words' => $totalAvailable,
                'words_needed' => $wordsNeeded,
                'page_type' => $pageType
            ], 402);
        }

        try {
            // Build the prompt based on generation type
            $prompt = $this->buildPrompt($title, $places, $generationType);
            Log::info('Prompt generated', [
                'type' => $generationType,
                'prompt_length' => strlen($prompt)
            ]);

            // Call Gemini API
            $apiResponse = $this->callGeminiApi($aiKey->gemini_api_key, $prompt);

            if (isset($apiResponse['error'])) {
                Log::error('Gemini API error', ['error' => $apiResponse['error']]);
                return response()->json([
                    'error' => 'AI service error',
                    'details' => $apiResponse['error']['message'] ?? 'Unknown error'
                ], 500);
            }

            $generatedText = $apiResponse['candidates'][0]['content']['parts'][0]['text'] ?? '';

            if (empty($generatedText)) {
                return response()->json([
                    'error' => 'No content generated. Please try again with a different title.'
                ], 500);
            }

            // Calculate word count and cost
            $wordCount = str_word_count(strip_tags($generatedText));
            
            // Calculate cost with proper top-up handling
            $costResult = $this->calculateCost($wordCount, $travelAgentId);
            
            // Check if there was an error in cost calculation
            if (isset($costResult['error'])) {
                return response()->json([
                    'error' => $costResult['error']
                ], 402);
            }
            
            $cost = $costResult['cost'];
            $usedFreeCredit = $costResult['used_free'];
            $usedTopUpWords = $costResult['used_top_up'] ?? 0;

            // Log the usage with correct parameters
            $this->logUsage($aiKey, $title, $prompt, $wordCount, $cost, $usedFreeCredit, $travelAgentId, $usedTopUpWords);

            // Refresh the AI key to get updated counts
            $aiKey->refresh();

            return response()->json([
                'description' => $generatedText,
                'word_count' => $wordCount,
                'cost' => $cost,
                'remaining_free_words' => $aiKey->getRemainingFreeWords(),
                'remaining_subscription_words' => $aiKey->getSubscriptionWords(),
                'remaining_top_up_words' => $aiKey->getRemainingTopUpWords(),
                'total_available_words' => $aiKey->getTotalAvailableWords(),
                'used_free_credit' => $usedFreeCredit,
                'used_top_up_words' => $usedTopUpWords,
                'has_active_subscription' => $aiKey->hasActiveSubscription(),
                'current_top_up_rate' => $aiKey->getCurrentTopUpRate()
            ]);

        } catch (\Exception $e) {
            Log::error('AI generation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'travel_agent_id' => $travelAgentId,
                'title' => $title,
                'type' => $generationType
            ]);

            return response()->json([
                'error' => 'AI generation failed. Please try again.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate complete trip plan with AI including practical requirements
     */
    public function generateTripPlan(Request $request)
    {
        // Validate inputs including start_point and end_point
        $validator = Validator::make($request->all(), [
            'destination' => 'required|string|max:255',
            'start_point' => 'required|string|max:255',
            'end_point' => 'required|string|max:255',
            'duration' => 'required|integer|min:2|max:15',
            'pax' => 'required|integer|min:1|max:50',
            'trip_type' => 'required|string|in:family,honeymoon,adventure,religious,business,couples,senior',
            'budget' => 'nullable|string|in:budget,standard,luxury',
            'travel_agent_id' => 'required|exists:travel_agents,id',
            'customer_id' => 'required|exists:customers,id',
            'start_date' => 'required|date',
            'tax_percentage' => 'nullable|numeric|min:0|max:100',
            'margin_percentage' => 'nullable|numeric|min:0|max:100', 
            'discount_amount' => 'nullable|numeric|min:0',
            'optimized_for_long_trip' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => 'Validation failed',
                'messages' => $validator->errors()
            ], 422);
        }

        // Get travel agent AI key
        $aiKey = AiKey::where('travel_agent_id', $request->travel_agent_id)->first();

        if (!$aiKey) {
            return response()->json([
                'error' => 'AI key not found for this travel agent'
            ], 404);
        }

        if (!$aiKey->is_active) {
            return response()->json([
                'error' => 'AI service is not active for your account'
            ], 403);
        }

        // Check word limits (scale with duration)
        $minWordsRequired = self::MIN_WORDS_ITINERARY * $request->duration;
        $totalAvailable = $aiKey->getTotalAvailableWords();
        
        if ($totalAvailable < $minWordsRequired) {
            $wordsNeeded = $minWordsRequired - $totalAvailable;
            return response()->json([
                'error' => "Insufficient credits. You need {$wordsNeeded} more words for {$request->duration}-day trip.",
                'required_words' => $minWordsRequired,
                'available_words' => $totalAvailable,
                'words_needed' => $wordsNeeded
            ], 402);
        }

        try {
            // 1. Gather available resources
            $availableData = $this->gatherAvailableResources($request->travel_agent_id);
            
            // 2. Build smart AI prompt with optimization for long trips
            $isLongTrip = $request->duration > 5 || $request->optimized_for_long_trip;
            $prompt = $this->buildTripPlanPrompt($request, $availableData, $isLongTrip);
            
            Log::info('AI Trip Plan Prompt', [
                'travel_agent_id' => $request->travel_agent_id,
                'destination' => $request->destination,
                'start_point' => $request->start_point,
                'end_point' => $request->end_point,
                'duration' => $request->duration,
                'is_long_trip' => $isLongTrip
            ]);

            // 3. Call AI with optimized settings for long trips
            $apiResponse = $this->callGeminiApiForTripPlan($aiKey->gemini_api_key, $prompt, $isLongTrip);
            
            if (isset($apiResponse['error'])) {
                throw new \Exception($apiResponse['error']['message'] ?? 'AI service error');
            }

            $generatedText = $apiResponse['candidates'][0]['content']['parts'][0]['text'] ?? '';
            
            if (empty($generatedText)) {
                throw new \Exception('No content generated by AI');
            }

            // 4. Parse and validate JSON response with enhanced handling
            $aiPlan = $this->parseAndValidateAiResponse($generatedText);
            
            // 5. Process and match resources with geographical intelligence
            $processedPlan = $this->processAiPlan($aiPlan, $availableData, $request->travel_agent_id, $request->start_date);
            
            // 6. Calculate complete costing using exact frontend logic
            $costing = $this->calculateCompleteCosting($processedPlan, [
                'tax_percentage' => $request->tax_percentage ?? 18,
                'margin_percentage' => $request->margin_percentage ?? 0, 
                'discount_amount' => $request->discount_amount ?? 0
            ]);
            
            // 7. Calculate word count and cost
            $wordCount = str_word_count(strip_tags($generatedText));
            $costResult = $this->calculateCost($wordCount, $request->travel_agent_id);
            
            if (isset($costResult['error'])) {
                return response()->json(['error' => $costResult['error']], 402);
            }
            
            // 8. Log usage
            $this->logUsage($aiKey, "Trip Plan: {$request->destination}", $prompt, $wordCount, 
                           $costResult['cost'], $costResult['used_free'], $request->travel_agent_id, 
                           $costResult['used_top_up'] ?? 0);

            // Refresh AI key to get updated counts
            $aiKey->refresh();

            return response()->json([
                'success' => true,
                'plan' => $processedPlan,
                'costing' => $costing,
                'word_count' => $wordCount,
                'cost' => $costResult['cost'],
                'remaining_words' => $aiKey->getTotalAvailableWords(),
                'is_long_trip' => $isLongTrip
            ]);

        } catch (\Exception $e) {
            Log::error('AI Trip Plan generation failed', [
                'error' => $e->getMessage(),
                'travel_agent_id' => $request->travel_agent_id,
                'destination' => $request->destination,
                'duration' => $request->duration
            ]);

            return response()->json([
                'error' => 'Trip plan generation failed: ' . $e->getMessage(),
                'suggestion' => $request->duration > 7 ? 'Try breaking your trip into shorter segments or reducing the number of destinations.' : 'Please try again with different parameters.'
            ], 500);
        }
    }

    /**
     * Gather all available resources for the travel agent
     */
    private function gatherAvailableResources($travelAgentId)
    {
        return [
            'services' => Service::where('travel_agent_id', $travelAgentId)
                ->get()
                ->toArray(),
                
            'hotels' => Hotel::where('travel_agent_id', $travelAgentId)
                ->get()
                ->toArray(),
                
            'extras' => ExtraService::where('travel_agent_id', $travelAgentId)
                ->get()
                ->toArray(),
                
            'day_descriptions' => DayDescription::where('travel_agent_id', $travelAgentId)
                ->get()
                ->toArray()
        ];
    }

    /**
     * Build intelligent trip plan prompt with optimization for long trips
     */
    private function buildTripPlanPrompt($request, $availableData, $isLongTrip = false)
    {
        // Parse multiple destinations
        $destinationInput = $request->destination;
        $destinations = array_map('trim', explode(',', $destinationInput));
        $destinationText = implode(' + ', $destinations);
        
        $startPoint = $request->start_point;
        $endPoint = $request->end_point;
        $duration = $request->duration;
        $pax = $request->pax;
        $tripType = $request->trip_type;
        $budget = $request->budget ?? 'standard';

        // Generate attractive package name with destination
        $attractivePackageName = $this->generateAttractivePackageName($destinations, $duration, $tripType, $startPoint, $endPoint);
        
        // Get route type for context
        $routeType = $this->getRouteType($startPoint, $endPoint, $destinations);

        // OPTIMIZED: Simplified geographical context to reduce token usage
        $geoContext = $this->getOptimizedGeoContext($startPoint, $endPoint, $destinations, $availableData);
        
        // OPTIMIZED: Build concise resources list
        $servicesList = $this->buildOptimizedServicesList($availableData['services']);
        $hotelsList = $this->buildOptimizedHotelsList($availableData['hotels']);
        $extrasList = array_column($availableData['extras'], 'name');

        $baseWordLimit = $isLongTrip ? 80 : 150;
        $longTripInstructions = $isLongTrip ? 
            "CRITICAL FOR LONG TRIP: Keep day descriptions VERY CONCISE (max {$baseWordLimit} words). Focus on essential logistics. Use bullet points for activities where possible." : 
            "Keep day descriptions concise (max {$baseWordLimit} words).";

        // *** START: CRITICAL PERMIT INSTRUCTION & TEXT ***
        $permitText = $this->getPermitRequirementsText();
        $permitInstruction = "
**CRITICAL PERMIT INSTRUCTION:** If a day's plan includes travel to restricted areas (e.g., **Lachung, Lachen, State Border Crossings**) or requires an **Inner Line Permit (ILP)**, you **MUST** append the following exact block of text to the end of that day's `day_description` field. The word count for this block is **EXEMPT** from the {$baseWordLimit}-word limit.

PERMIT BLOCK TO INJECT:
" . trim($permitText) . "
";
        // *** END: CRITICAL PERMIT INSTRUCTION & TEXT ***

        return "You are an expert Northeast India travel planner. Create a detailed $duration-day $tripType trip starting from $startPoint, covering $destinationText, and ending at $endPoint for $pax people ($budget budget).

$longTripInstructions

$permitInstruction

ROUTE TYPE: $routeType
DURATION: $duration days
DESTINATIONS: " . implode(' → ', $destinations) . "

GEOGRAPHICAL CONTEXT:
- Route: " . implode(' → ', $geoContext['suggested_route']) . "
- Travel Considerations: " . implode(', ', $geoContext['key_considerations']) . "

AVAILABLE RESOURCES:

SERVICES (Use EXACT variation names):
$servicesList

HOTELS (Use EXACT room type names):
$hotelsList

EXTRA SERVICES:
" . implode(', ', $extrasList) . "

CRITICAL JSON RULES:
1. Return ONLY pure JSON without any surrounding text
2. Use ONLY exact variation/room types listed above
3. Ensure all brackets and braces are properly closed
4. Remove any trailing commas
5. Keep descriptions concise to avoid truncation

RETURN ONLY VALID JSON with this exact structure:
{
   \"package_name\": \"$attractivePackageName\",
   \"trip_days\": [
     {
       \"day_number\": 1,
       \"day_title\": \"Concise day title\",
       \"services\": [
         {\"service_name\": \"EXACT_MATCH\", \"variation\": \"Medium\", \"qty\": 1}
       ],
       \"hotels\": [
         {\"hotel_name\": \"EXACT_MATCH\", \"room_type\": \"Standard\", \"rooms\": 2}
       ],
       \"extra_services\": [
         {\"extra_name\": \"EXACT_MATCH\", \"quantity\": 1}
       ],
       \"day_description\": \"Brief description focusing on key activities and travel. " . ($isLongTrip ? "Maximum 80 words." : "Maximum 150 words.") . "\"
     }
   ]
}";
    }

    /**
     * Optimized geographical context to reduce token usage
     */
    private function getOptimizedGeoContext($startPoint, $endPoint, $destinations, $availableData)
    {
        $suggestedRoute = [];
        $suggestedRoute[] = $startPoint;
        $suggestedRoute = array_merge($suggestedRoute, $destinations);
        $suggestedRoute[] = $endPoint;
        
        $keyConsiderations = [];
        
        // Add only essential considerations
        if (str_contains(strtolower($startPoint), 'kolkata')) {
            $keyConsiderations[] = 'Overnight train/flight from Kolkata recommended';
        }
        
        if (in_array('lachung', array_map('strtolower', $destinations)) || 
            in_array('lachen', array_map('strtolower', $destinations))) {
            $keyConsiderations[] = 'Permit required for Lachung/Lachen';
            $keyConsiderations[] = 'Mountain roads - plan extra travel time';
        }
        
        if (count($destinations) > 2) {
            $keyConsiderations[] = 'Multi-destination - optimize travel days';
        }

        return [
            'suggested_route' => $suggestedRoute,
            'key_considerations' => $keyConsiderations
        ];
    }

    /**
     * Build optimized services list
     */
    private function buildOptimizedServicesList($services)
    {
        $servicesList = [];
        foreach ($services as $service) {
            $variations = array_column($service['type_variation'], 'type');
            $servicesList[] = $service['name'] . ' [' . implode(', ', $variations) . ']';
        }
        return implode('; ', $servicesList);
    }

    /**
     * Build optimized hotels list
     */
    private function buildOptimizedHotelsList($hotels)
    {
        $hotelsList = [];
        foreach ($hotels as $hotel) {
            $variations = array_column($hotel['room_variations'], 'type');
            $hotelsList[] = $hotel['name'] . ' [' . implode(', ', $variations) . ']';
        }
        return implode('; ', $hotelsList);
    }

    /**
     * Generate attractive package names with destination names
     */
    private function generateAttractivePackageName($destinations, $duration, $tripType, $startPoint, $endPoint)
    {
        $mainDestination = !empty($destinations) ? $destinations[0] : 'Northeast India';
        $tripTypeDisplay = ucfirst($tripType);
        
        $packageTemplates = [
            // Multi-destination templates
            "{$duration}-Day {$tripTypeDisplay} Tour: " . implode(' & ', $destinations),
            "{$duration} Days in Paradise: " . implode(' + ', $destinations),
            "{$tripTypeDisplay} Getaway: {$duration} Days Exploring " . implode(', ', $destinations),
            "{$duration}-Day {$tripTypeDisplay} Circuit: " . implode(' → ', $destinations),
            "{$mainDestination} & Beyond: {$duration}-Day {$tripTypeDisplay} Adventure",
            
            // Single destination templates
            "{$duration}-Day {$tripTypeDisplay} Tour of {$mainDestination}",
            "{$mainDestination} Dream Vacation: {$duration} Days of Bliss",
            "{$tripTypeDisplay} Escape to {$mainDestination} - {$duration} Days",
            "{$duration}-Day {$mainDestination} {$tripTypeDisplay} Package",
            "Discover {$mainDestination}: {$duration}-Day {$tripTypeDisplay} Journey",
        ];
        
        // Choose template based on trip type and number of destinations
        if (count($destinations) > 1) {
            return $packageTemplates[array_rand(array_slice($packageTemplates, 0, 5))];
        } else {
            return $packageTemplates[array_rand(array_slice($packageTemplates, 5, 5))];
        }
    }

    /**
     * Call Gemini API specifically for trip planning with optimization for long trips
     */
    private function callGeminiApiForTripPlan($encryptedApiKey, $prompt, $isLongTrip = false)
    {
        try {
            $apiKey = Crypt::decrypt($encryptedApiKey);
            
            $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key={$apiKey}";
            
            $response = Http::timeout($isLongTrip ? 120 : 60) // Longer timeout for long trips
                ->withoutVerifying()
                ->withHeaders([
                    'Content-Type' => 'application/json',
                ])->post($url, [
                    'contents' => [
                        [
                            'parts' => [
                                ['text' => $prompt]
                            ]
                        ]
                    ],
                    'generationConfig' => [
                        'temperature' => 0.7,
                        'topK' => 40,
                        'topP' => 0.9,
                        'maxOutputTokens' => $isLongTrip ? 8192 : 4096, // Increased for long trips
                        'responseMimeType' => 'application/json',
                    ]
                ]);

            if ($response->failed()) {
                Log::error('Gemini API failed for trip planning', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                
                throw new \Exception("API request failed with status: " . $response->status());
            }

            $responseData = $response->json();

            // Log the raw response for debugging
            Log::info('Gemini API response received', [
                'response_length' => strlen(json_encode($responseData)),
                'has_candidates' => isset($responseData['candidates']),
                'candidate_count' => isset($responseData['candidates']) ? count($responseData['candidates']) : 0,
                'is_long_trip' => $isLongTrip
            ]);

            return $responseData;

        } catch (\Exception $e) {
            Log::error('Gemini API call exception in trip planning', [
                'message' => $e->getMessage(),
                'is_long_trip' => $isLongTrip
            ]);
            throw $e;
        }
    }

    /**
     * Enhanced AI response parsing with better error handling
     */
    private function parseAndValidateAiResponse($generatedText)
    {
        try {
            // More aggressive cleaning of the response
            $cleanedText = $this->cleanAiResponse($generatedText);
            
            Log::info('Cleaned AI response for parsing', [
                'original_length' => strlen($generatedText),
                'cleaned_length' => strlen($cleanedText)
            ]);

            // First attempt: direct JSON decode
            $aiPlan = json_decode($cleanedText, true);
            
            if (json_last_error() === JSON_ERROR_NONE && isset($aiPlan['package_name']) && isset($aiPlan['trip_days'])) {
                Log::info('✅ AI response parsed successfully on first attempt');
                return $aiPlan;
            }

            // Second attempt: Try to fix common JSON issues
            $fixedText = $this->fixCommonJsonIssues($cleanedText);
            $aiPlan = json_decode($fixedText, true);
            
            if (json_last_error() === JSON_ERROR_NONE && isset($aiPlan['package_name']) && isset($aiPlan['trip_days'])) {
                Log::info('✅ AI response parsed successfully after fixing common issues');
                return $aiPlan;
            }

            // Third attempt: Extract and validate JSON object
            $jsonMatch = $this->extractAndValidateJson($cleanedText);
            if ($jsonMatch) {
                $aiPlan = json_decode($jsonMatch, true);
                if (json_last_error() === JSON_ERROR_NONE && isset($aiPlan['package_name']) && isset($aiPlan['trip_days'])) {
                    Log::info('✅ AI response parsed successfully after extraction');
                    return $aiPlan;
                }
            }

            // Final attempt: Try to salvage partial response
            $salvagedPlan = $this->salvagePartialResponse($cleanedText);
            if ($salvagedPlan) {
                Log::info('✅ Partial AI response salvaged successfully');
                return $salvagedPlan;
            }

            throw new \Exception('Invalid JSON response from AI after all attempts: ' . json_last_error_msg());

        } catch (\Exception $e) {
            Log::error('AI response parsing failed', [
                'error' => $e->getMessage(),
                'json_error' => json_last_error_msg(),
                'response_sample' => substr($generatedText, 0, 500) . '...' . substr($generatedText, -200)
            ]);
            throw new \Exception('Failed to parse AI response: ' . $e->getMessage());
        }
    }

    /**
     * Clean AI response with multiple strategies
     */
    private function cleanAiResponse($text)
    {
        // Remove markdown code blocks
        $cleaned = preg_replace('/```json\s*/', '', $text);
        $cleaned = preg_replace('/```\s*$/', '', $cleaned);
        $cleaned = trim($cleaned);
        
        // Remove any BOM characters
        $cleaned = preg_replace('/\x{FEFF}/u', '', $cleaned);
        
        // Remove invalid control characters (except newlines and tabs)
        $cleaned = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $cleaned);
        
        return $cleaned;
    }

    /**
     * Fix common JSON issues in AI responses
     */
    private function fixCommonJsonIssues($jsonString)
    {
        // Remove any content before first { and after last }
        $startPos = strpos($jsonString, '{');
        $endPos = strrpos($jsonString, '}');
        
        if ($startPos !== false && $endPos !== false) {
            $jsonString = substr($jsonString, $startPos, $endPos - $startPos + 1);
        }

        // Fix trailing commas in arrays and objects
        $jsonString = preg_replace('/,\s*([\]}])/m', '$1', $jsonString);
        
        // Fix unescaped quotes in strings
        $jsonString = preg_replace_callback('/"([^"\\\\]*(?:\\\\.[^"\\\\]*)*)"|\'([^\'\\\\]*(?:\\\\.[^\'\\\\]*)*)\'/s', 
            function($matches) {
                if (isset($matches[1])) {
                    return '"' . addslashes($matches[1]) . '"';
                }
                return '"' . addslashes($matches[2]) . '"';
            }, 
            $jsonString
        );

        // Fix missing quotes around keys
        $jsonString = preg_replace('/(\w+):/', '"$1":', $jsonString);

        return $jsonString;
    }

    /**
     * Extract and validate JSON from text
     */
    private function extractAndValidateJson($text)
    {
        // Try to find the main JSON object
        $depth = 0;
        $startPos = strpos($text, '{');
        if ($startPos === false) return null;

        $jsonContent = '';
        $inString = false;
        $escapeNext = false;

        for ($i = $startPos; $i < strlen($text); $i++) {
            $char = $text[$i];

            if ($escapeNext) {
                $jsonContent .= $char;
                $escapeNext = false;
                continue;
            }

            if ($char === '\\') {
                $jsonContent .= $char;
                $escapeNext = true;
                continue;
            }

            if ($char === '"') {
                $inString = !$inString;
            }

            if (!$inString) {
                if ($char === '{') {
                    $depth++;
                } elseif ($char === '}') {
                    $depth--;
                }
            }

            $jsonContent .= $char;

            if ($depth === 0 && $i > $startPos) {
                // We've found a complete JSON object
                break;
            }
        }

        return $depth === 0 ? $jsonContent : null;
    }

    /**
     * Salvage partial response when JSON is incomplete
     */
    private function salvagePartialResponse($text)
    {
        try {
            $startPos = strpos($text, '{');
            if ($startPos === false) return null;

            // Extract what we have
            $partialJson = substr($text, $startPos);
            
            // Try to complete the JSON if it's truncated
            if (strpos($partialJson, '"trip_days"') !== false) {
                // Find the trip_days array and ensure it's closed
                $tripDaysPos = strpos($partialJson, '"trip_days"');
                $arrayStart = strpos($partialJson, '[', $tripDaysPos);
                
                if ($arrayStart !== false) {
                    $bracketCount = 1;
                    $i = $arrayStart + 1;
                    
                    while ($i < strlen($partialJson) && $bracketCount > 0) {
                        $char = $partialJson[$i];
                        if ($char === '[') $bracketCount++;
                        if ($char === ']') $bracketCount--;
                        $i++;
                    }
                    
                    // If we found the end of trip_days array, close the JSON properly
                    if ($bracketCount === 0) {
                        $completeJson = substr($partialJson, 0, $i) . '}';
                        $aiPlan = json_decode($completeJson, true);
                        
                        if (json_last_error() === JSON_ERROR_NONE && isset($aiPlan['package_name']) && isset($aiPlan['trip_days'])) {
                            Log::warning('Salvaged partial AI response', [
                                'original_length' => strlen($text),
                                'salvaged_length' => strlen($completeJson)
                            ]);
                            return $aiPlan;
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            Log::error('Failed to salvage partial response', ['error' => $e->getMessage()]);
        }
        
        return null;
    }

    /**
     * Process AI plan and match with actual database resources
     */
    private function processAiPlan($aiPlan, $availableData, $travelAgentId, $startDate)
    {
        $processedDays = [];
        $currentDate = new \DateTime($startDate);
        
        foreach ($aiPlan['trip_days'] as $index => $aiDay) {
            $processedDay = [
                'day_number' => $index + 1,
                'day_title' => $aiDay['day_title'] ?? 'Day ' . ($index + 1),
                'date' => $currentDate->format('Y-m-d'),
                'services' => $this->matchServices($aiDay['services'] ?? [], $availableData['services']),
                'hotels' => $this->matchHotels($aiDay['hotels'] ?? [], $availableData['hotels']),
                'extra_services' => $this->matchExtraServices($aiDay['extra_services'] ?? [], $availableData['extras']),
                'day_description' => $aiDay['day_description'] ?? ''
            ];
            
            $processedDays[] = $processedDay;
            $currentDate->modify('+1 day');
        }
        
        return [
            'package_name' => $aiPlan['package_name'],
            'trip_days' => $processedDays
        ];
    }

    /**
     * Match AI-suggested services with actual database services
     */
    private function matchServices($aiServices, $availableServices)
    {
        $matchedServices = [];
        
        foreach ($aiServices as $aiService) {
            $serviceName = $aiService['service_name'];
            $aiVariation = $aiService['variation'] ?? 'Standard';
            $matchedService = null;
            
            // Exact match for service name
            foreach ($availableServices as $service) {
                if (strcasecmp(trim($service['name']), trim($serviceName)) === 0) {
                    $matchedService = $service;
                    break;
                }
            }
            
            // Fuzzy match if exact not found
            if (!$matchedService) {
                $matchedService = $this->fuzzyMatchService($serviceName, $availableServices);
            }
            
            if ($matchedService) {
                // Validate variation exists
                $validVariations = array_column($matchedService['type_variation'], 'type');
                $validVariation = $this->validateVariation($aiVariation, $validVariations);
                
                $matchedServices[] = [
                    'service_id' => $matchedService['id'],
                    'service_name' => $matchedService['name'],
                    'variation' => $validVariation,
                    'qty' => $aiService['qty'] ?? 1,
                    'type_variation' => $matchedService['type_variation'] ?? []
                ];
            } else {
                // Service not found in database - skip it
                Log::warning("Service not found in database: " . $serviceName);
            }
        }
        
        return $matchedServices;
    }

    private function matchHotels($aiHotels, $availableHotels)
    {
        $matchedHotels = [];
        
        foreach ($aiHotels as $aiHotel) {
            $hotelName = $aiHotel['hotel_name'];
            $aiRoomType = $aiHotel['room_type'] ?? 'Standard';
            $matchedHotel = null;
            
            // Exact match for hotel name
            foreach ($availableHotels as $hotel) {
                if (strcasecmp(trim($hotel['name']), trim($hotelName)) === 0) {
                    $matchedHotel = $hotel;
                    break;
                }
            }
            
            // Fuzzy match if exact not found
            if (!$matchedHotel) {
                $matchedHotel = $this->fuzzyMatchHotel($hotelName, $availableHotels);
            }
            
            if ($matchedHotel) {
                // Validate room type exists
                $validRoomTypes = array_column($matchedHotel['room_variations'], 'type');
                $validRoomType = $this->validateVariation($aiRoomType, $validRoomTypes);
                
                $matchedHotels[] = [
                    'hotel_id' => $matchedHotel['id'],
                    'hotel_name' => $matchedHotel['name'],
                    'room_type' => $validRoomType,
                    'rooms' => $aiHotel['rooms'] ?? 1,
                    'room_variations' => $matchedHotel['room_variations'] ?? []
                ];
            } else {
                // Hotel not found in database - skip it
                Log::warning("Hotel not found in database: " . $hotelName);
            }
        }
        
        return $matchedHotels;
    }

    private function validateVariation($aiVariation, $validVariations)
    {
        // Clean the AI variation (remove extra spaces, convert to lowercase for comparison)
        $cleanAiVariation = trim(strtolower($aiVariation));
        
        // First try exact match
        foreach ($validVariations as $valid) {
            if (strcasecmp(trim($valid), trim($aiVariation)) === 0) {
                return $valid; // Return the exact database variation
            }
        }
        
        // Try partial match for common variations
        foreach ($validVariations as $valid) {
            $cleanValid = trim(strtolower($valid));
            
            // Check for common patterns
            if (str_contains($cleanAiVariation, 'standard') && str_contains($cleanValid, 'standard')) {
                return $valid;
            }
            if (str_contains($cleanAiVariation, 'luxury') && str_contains($cleanValid, 'luxury')) {
                return $valid;
            }
            if (str_contains($cleanAiVariation, 'deluxe') && str_contains($cleanValid, 'deluxe')) {
                return $valid;
            }
            if (str_contains($cleanAiVariation, 'small') && str_contains($cleanValid, 'small')) {
                return $valid;
            }
            if (str_contains($cleanAiVariation, 'medium') && str_contains($cleanValid, 'medium')) {
                return $valid;
            }
            if (str_contains($cleanAiVariation, 'large') && str_contains($cleanValid, 'large')) {
                return $valid;
            }
        }
        
        // If no match found, use the first valid variation
        Log::warning("Variation '$aiVariation' not found, using first available: " . ($validVariations[0] ?? 'Standard'));
        return $validVariations[0] ?? 'Standard';
    }

    private function fuzzyMatchService($serviceName, $availableServices)
    {
        $bestMatch = null;
        $bestScore = 0;
        
        foreach ($availableServices as $service) {
            similar_text(
                strtolower(trim($serviceName)),
                strtolower(trim($service['name'])),
                $score
            );
            
            if ($score > $bestScore && $score > 70) {
                $bestScore = $score;
                $bestMatch = $service;
            }
        }
        
        return $bestMatch;
    }

    private function fuzzyMatchHotel($hotelName, $availableHotels)
    {
        $bestMatch = null;
        $bestScore = 0;
        
        foreach ($availableHotels as $hotel) {
            similar_text(
                strtolower(trim($hotelName)),
                strtolower(trim($hotel['name'])),
                $score
            );
            
            if ($score > $bestScore && $score > 70) {
                $bestScore = $score;
                $bestMatch = $hotel;
            }
        }
        
        return $bestMatch;
    }

    /**
     * Match AI-suggested extra services
     */
    private function matchExtraServices($aiExtras, $availableExtras)
    {
        $matchedExtras = [];
        
        foreach ($aiExtras as $aiExtra) {
            $extraName = $aiExtra['extra_name'];
            $matchedExtra = null;
            
            foreach ($availableExtras as $extra) {
                if (strcasecmp(trim($extra['name']), trim($extraName)) === 0) {
                    $matchedExtra = $extra;
                    break;
                }
            }
            
            if ($matchedExtra) {
                $matchedExtras[] = [
                    'extra_service_id' => $matchedExtra['id'],
                    'extra_name' => $matchedExtra['name'],
                    'quantity' => $aiExtra['quantity'] ?? 1,
                    'price' => $matchedExtra['price'] ?? 0
                ];
            } else {
                // Extra service not found - will be created later
                $matchedExtras[] = [
                    'extra_service_id' => null,
                    'extra_name' => $extraName,
                    'quantity' => $aiExtra['quantity'] ?? 1,
                    'price' => 0
                ];
            }
        }
        
        return $matchedExtras;
    }

    /**
     * Returns the standardised permit and document requirements text block for embedding in descriptions.
     */
    private function getPermitRequirementsText()
    {
        // Use a clear, unique delimiter (e.g., ###PERMIT_DOCS_START###) for the AI to find and include.
        return "\n\n###PERMIT_DOCS_START###\n***IMPORTANT: PERMIT REQUIRED***\nThis day requires an Inner Line Permit (ILP) or special permission/border crossing. Traveller must arrange the following documents:\n- Valid Photo Identity Card (Voter Card/Driving License/Passport/AADHAR Card) for each adult (Original and 04 Photocopies)\n- Valid Birth Certificate or AADHAR Card for age under 18 years (Original and 04 Photocopies)\n- Need 04 copies of Passport size photo of each person including children/kids\n###PERMIT_DOCS_END###\n";
    }

    /**
     * Calculate complete costing using exact frontend logic
     */
    private function calculateCompleteCosting($processedPlan, $costingParams)
    {
        $taxPercentage = $costingParams['tax_percentage'];
        $marginPercentage = $costingParams['margin_percentage'];
        $discountAmount = $costingParams['discount_amount'];
        
        // Calculate billed amount (same as your calculateBilledAmount())
        $billedAmount = $this->calculateBilledAmount($processedPlan);
        
        // Calculate margin amount (same as your calculateMarginAmount())
        $marginAmount = round($billedAmount * $marginPercentage / 100, 2);
        
        // Calculate tax amount (same as your calculateTaxAmount())
        $subtotalWithMargin = $billedAmount + $marginAmount;
        $taxAmount = round($subtotalWithMargin * $taxPercentage / 100, 2);
        
        // Calculate final price (same as your calculateFinalPrice())
        $finalPrice = round($billedAmount + $marginAmount + $taxAmount - $discountAmount, 2);
        
        return [
            'billed_amount' => $billedAmount,
            'margin_amount' => $marginAmount,
            'tax_amount' => $taxAmount,
            'discount_amount' => $discountAmount,
            'final_price' => $finalPrice,
            'tax_percentage' => $taxPercentage,
            'margin_percentage' => $marginPercentage
        ];
    }

    /**
     * Calculate billed amount - matches your frontend logic exactly
     */
    private function calculateBilledAmount($processedPlan)
    {
        $subtotal = 0;
        
        foreach ($processedPlan['trip_days'] as $day) {
            // Services calculation
            foreach ($day['services'] as $service) {
                foreach ($service['type_variation'] as $variation) {
                    if ($variation['type'] === $service['variation']) {
                        $subtotal += $variation['price'] * $service['qty'];
                        break;
                    }
                }
            }
            
            // Hotels calculation  
            foreach ($day['hotels'] as $hotel) {
                foreach ($hotel['room_variations'] as $variation) {
                    if ($variation['type'] === $hotel['room_type']) {
                        $subtotal += $variation['price'] * $hotel['rooms'];
                        break;
                    }
                }
            }
            
            // Extra services calculation
            foreach ($day['extra_services'] as $extra) {
                $subtotal += $extra['price'] * $extra['quantity'];
            }
        }
        
        return round($subtotal, 2);
    }

    // ========== EXISTING METHODS ==========

    private function getPageTypeName($minWords)
    {
        switch ($minWords) {
            case self::MIN_WORDS_DESCRIPTION:
                return 'description';
            case self::MIN_WORDS_ITINERARY:
                return 'itinerary';
            default:
                return 'content';
        }
    }

    private function callGeminiApi($encryptedApiKey, $prompt)
    {
        try {
            // Decrypt the Laravel-encrypted API key
            $apiKey = Crypt::decrypt($encryptedApiKey);
            
            $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key={$apiKey}";
            
            Log::info('Calling Gemini API from XAMPP localhost', [
                'url' => $url,
                'api_key_first_10' => substr($apiKey, 0, 10) . '...'
            ]);

            // Disable SSL verification for XAMPP local development
            $response = Http::timeout(300)
                ->withoutVerifying()
                ->withHeaders([
                    'Content-Type' => 'application/json',
                ])->post($url, [
                    'contents' => [
                        [
                            'parts' => [
                                ['text' => $prompt]
                            ]
                        ]
                    ],
                    'generationConfig' => [
                        'temperature' => 0.7,
                        'topK' => 40,
                        'topP' => 0.95,
                        'maxOutputTokens' => 1024,
                    ]
                ]);

            Log::info('Gemini API response status', ['status' => $response->status()]);

            if ($response->failed()) {
                Log::error('Gemini API failed', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'error' => $response->json()
                ]);
                
                throw new \Exception("API request failed with status: " . $response->status());
            }

            return $response->json();

        } catch (\Exception $e) {
            Log::error('Gemini API call exception', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    private function buildPrompt($title, $places = null, $generationType = 'description')
    {
        $lowerTitle = strtolower($title);
        $hasPlaces = !empty($places);

        if (str_contains($lowerTitle, 'sightseeing') || str_contains($lowerTitle, 'tour') || $hasPlaces) {
            return "Generate a professional travel sightseeing description for: \"$title\"

Follow these instructions exactly:

1. Create a section titled **Places of Interest:**. Under this title, generate a bulleted list (using dashes '-') of key attractions.
" . ($hasPlaces ? "2. MUST INCLUDE these specific places provided by the user: $places" : "2. Include 4-6 key attractions relevant to the title.") . "
3. Create a second section titled **Tour Description:**. Under this title, write a VERY CONCISE description of the overall experience.
4. Keep the entire description SHORT and COMPACT - maximum 100 words total.
5. Use bullet points only for Places of Interest, keep the description paragraph brief.

Focus on being brief yet informative. Include specific place names when provided.";
        }
        else if (str_contains($lowerTitle, 'airport') || str_contains($lowerTitle, 'bagdogra') || 
                 str_contains($lowerTitle, 'njp') || str_contains($lowerTitle, 'railway') || 
                 str_contains($lowerTitle, 'bus') || str_contains($lowerTitle, 'siliguri junction') ||
                 str_contains($lowerTitle, 'transfer') || str_contains($lowerTitle, 'to ') || 
                 str_contains($lowerTitle, 'from ')) {
            
            $locationSpecific = '';
            if (str_contains($lowerTitle, 'siliguri')) {
                $locationSpecific = 'New Jalpaiguri Railway Station (NJP)/Siliguri Junction Bus Stand/Bagdogra Airport or'.$title;
            }

            return "Generate a professional travel transfer description for: \"$title\"

Create a VERY CONCISE description with these instructions:

1. **Route:** " . ($locationSpecific ?: 'Mention key pickup/drop points') . "
2. **Key Points:** Briefly mention scenic beauty, travel time, and highlights
3. **Style:** Write in a single, compact paragraph
4. **Length:** Maximum 80 words - make it as short as possible while covering essential points
5. **Focus:** Be specific about locations but extremely brief

Write the absolute shortest description possible that still covers the journey essentials.";
        }
        else {
            return "Generate a professional travel day description for: \"$title\"

Create a VERY SHORT description with these instructions:

1. **Content:** Brief overview of the travel experience
2. **Style:** Single compact paragraph
3. **Length:** Maximum 70 words - make it as concise as possible
4. **Focus:** Highlight only the most essential aspects

Write the shortest possible description that still conveys the main experience.";
        }
    }

    private function buildItineraryPrompt($title, $places = null)
    {
        $hasPlaces = !empty($places);

        return "Generate a comprehensive travel itinerary for: \"$title\"

Follow these instructions exactly:

1. Create a detailed day-by-day itinerary with specific timings
2. Include morning, afternoon, and evening activities for each day
3. Include transportation details between locations
4. Suggest meal times and restaurant options
5. Include accommodation suggestions
" . ($hasPlaces ? "6. MUST INCLUDE these specific places: $places" : "6. Include popular attractions and activities") . "
7. Keep the itinerary practical and realistic
8. Include some free time for relaxation
9. Provide estimated costs where relevant

Make the itinerary comprehensive yet easy to follow. Focus on creating a memorable travel experience.";
    }

    /**
     * Calculate cost with proper word usage order
     */
    private function calculateCost($wordCount, $travelAgentId)
    {
        $aiKey = AiKey::where('travel_agent_id', $travelAgentId)->first();
        
        if (!$aiKey) {
            return ['cost' => 0, 'used_free' => true, 'used_top_up' => 0];
        }

        $remainingFreeWords = $aiKey->getRemainingFreeWords();
        $remainingSubscriptionWords = $aiKey->getSubscriptionWords();
        $remainingTopUpWords = $aiKey->getRemainingTopUpWords();

        Log::info('Word calculation', [
            'word_count' => $wordCount,
            'free_remaining' => $remainingFreeWords,
            'subscription_remaining' => $remainingSubscriptionWords,
            'top_up_remaining' => $remainingTopUpWords
        ]);

        // 1. Use FREE words first
        $freeWordsUsed = min($wordCount, $remainingFreeWords);
        $wordsAfterFree = $wordCount - $freeWordsUsed;

        if ($wordsAfterFree == 0) {
            return ['cost' => 0, 'used_free' => true, 'used_top_up' => 0];
        }

        // 2. Use SUBSCRIPTION words second
        $subscriptionWordsUsed = min($wordsAfterFree, $remainingSubscriptionWords);
        $wordsAfterSubscription = $wordsAfterFree - $subscriptionWordsUsed;

        if ($wordsAfterSubscription == 0) {
            return ['cost' => 0, 'used_free' => false, 'used_top_up' => 0];
        }

        // 3. Use TOP-UP words third
        $topUpWordsUsed = min($wordsAfterSubscription, $remainingTopUpWords);
        $wordsAfterTopUp = $wordsAfterSubscription - $topUpWordsUsed;

        if ($wordsAfterTopUp == 0) {
            return [
                'cost' => 0, 
                'used_free' => false, 
                'used_top_up' => $topUpWordsUsed
            ];
        }

        // 4. If still words remain, user doesn't have enough credits
        return [
            'error' => "Insufficient credits during calculation. This should not happen with pre-check.",
            'cost' => 0,
            'used_free' => false,
            'used_top_up' => 0
        ];
    }

    /**
     * Log usage with proper parameter order and field names
     */
    private function logUsage($aiKey, $title, $prompt, $wordCount, $cost, $usedFreeCredit, $travelAgentId, $usedTopUpWords = 0)
    {
        try {
            // Update word counts with proper tracking
            $aiKey->incrementWordUsage($wordCount, $usedFreeCredit, $usedTopUpWords);

            // Create usage log
            AiUsageLog::create([
                'travel_agent_id' => $travelAgentId,
                'ai_key_id' => $aiKey->id,
                'title' => $title,
                'prompt_used' => $prompt,
                'word_count' => $wordCount,
                'used_free_credit' => $usedFreeCredit,
                'used_top_up_words' => $usedTopUpWords,
                'cost' => $cost
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to log AI usage', [
                'error' => $e->getMessage(),
                'travel_agent_id' => $travelAgentId
            ]);
        }
    }

    public function getUsageStats(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'travel_agent_id' => 'required|exists:travel_agents,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => 'Validation failed',
                'messages' => $validator->errors()
            ], 422);
        }

        $travelAgentId = $request->travel_agent_id;
        $aiKey = AiKey::where('travel_agent_id', $travelAgentId)->first();

        if (!$aiKey) {
            return response()->json([
                'error' => 'AI key not found for this travel agent'
            ], 404);
        }

        // Get current subscription
        $subscription = AiSubscription::where('travel_agent_id', $travelAgentId)
            ->where('status', 'active')
            ->with('plan')
            ->first();

        // Calculate usage stats with proper top-up tracking
        $freeWordsUsed = $aiKey->free_words_used ?? 0;
        $freeWordsRemaining = $aiKey->getRemainingFreeWords();
        
        $subscriptionWordsUsed = $subscription ? $subscription->words_used : 0;
        $subscriptionWordsRemaining = $subscription ? $subscription->words_remaining : 0;
        $subscriptionTotalWords = $subscription ? $subscription->plan->monthly_words : 0;
        
        $topUpWordsTotal = $aiKey->top_up_words ?? 0;
        $topUpWordsUsed = $aiKey->paid_words_used ?? 0;
        $topUpWordsRemaining = $aiKey->getRemainingTopUpWords();
        
        $totalWordsUsed = $aiKey->total_words_used ?? 0;

        // Get current plan
        $currentPlan = $subscription ? $subscription->plan : AiPlan::where('name', 'free')->first();

        return response()->json([
            'free_words' => [
                'used' => $freeWordsUsed,
                'remaining' => $freeWordsRemaining,
                'total' => 500
            ],
            'subscription_words' => [
                'used' => $subscriptionWordsUsed,
                'remaining' => $subscriptionWordsRemaining,
                'total' => $subscriptionTotalWords
            ],
            'top_up_words' => [
                'used' => $topUpWordsUsed,
                'remaining' => $topUpWordsRemaining,
                'total' => $topUpWordsTotal
            ],
            'total_words_used' => $totalWordsUsed,
            'total_available_words' => $aiKey->getTotalAvailableWords(),
            'current_plan' => $currentPlan,
            'has_active_subscription' => (bool) $subscription,
            'subscription_end_date' => $subscription ? $subscription->end_date : null,
            'current_top_up_rate' => $aiKey->getCurrentTopUpRate(),
            'minimum_requirements' => [
                'description' => self::MIN_WORDS_DESCRIPTION,
                'itinerary' => self::MIN_WORDS_ITINERARY
            ]
        ]);
    }

    // Add test method for debugging
    public function testConnection(Request $request)
    {
        try {
            $travelAgentId = $request->travel_agent_id ?? 1;
            $aiKey = AiKey::where('travel_agent_id', $travelAgentId)->first();
            
            if (!$aiKey || empty($aiKey->gemini_api_key)) {
                return response()->json(['error' => 'No API key found'], 404);
            }
            
            // Decrypt the API key for testing
            $decryptedKey = Crypt::decrypt($aiKey->gemini_api_key);
            
            // Test basic HTTPS connectivity
            $googleTest = Http::withoutVerifying()->timeout(10)->get('https://www.google.com');
            
            // Test Gemini API with decrypted key
            $geminiTest = Http::withoutVerifying()->timeout(30)->post(
                'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key=' . $decryptedKey,
                [
                    'contents' => [
                        [
                            'parts' => [
                                ['text' => 'Hello, test connection from XAMPP']
                            ]
                        ]
                    ]
                ]
            );
            
            return response()->json([
                'google_connectivity' => $googleTest->status(),
                'gemini_connectivity' => $geminiTest->status(),
                'gemini_response' => $geminiTest->json(),
                'api_key_set' => !empty($aiKey->gemini_api_key),
                'api_key_valid' => $aiKey->hasValidApiKey()
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'hint' => 'Check XAMPP PHP extensions: curl, openssl'
            ], 500);
        }
    }

    /**
     * Get route type classification
     */
    private function getRouteType($startPoint, $endPoint, $destinations)
    {
        $startLower = strtolower($startPoint);
        $endLower = strtolower($endPoint);
        
        if ($startLower === $endLower) {
            return 'Round Trip';
        }
        
        $northeastPoints = ['siliguri', 'njp', 'bagdogra', 'darjeeling', 'gangtok', 'kalimpong', 'pelling', 'lachung', 'lachen', 'mirik', 'ravangla'];
        $plainsPoints = ['kolkata', 'alipurduar', 'guwahati'];
        
        $startIsNortheast = in_array($startLower, $northeastPoints);
        $endIsNortheast = in_array($endLower, $northeastPoints);
        
        if (!$startIsNortheast && $endIsNortheast) {
            return 'Plains to Mountains Journey';
        } elseif ($startIsNortheast && !$endIsNortheast) {
            return 'Mountains to Plains Journey';
        } elseif (!$startIsNortheast && !$endIsNortheast) {
            return 'Cross-Region Journey';
        } else {
            return 'Northeast India Circuit';
        }
    }
}