<?php

namespace App\Filament\Resources\TravelAgents;

use App\Filament\Resources\TravelAgents\Pages;
use App\Models\TravelAgent;
use Filament\Actions; 
use Filament\Schemas\Schema;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use UnitEnum;
use BackedEnum;

class TravelAgentResource extends Resource
{
    protected static ?string $model = TravelAgent::class;

    protected static BackedEnum|string|null $navigationIcon = 'heroicon-o-building-office';

    protected static ?string $navigationLabel = 'Travel Agents';

    protected static ?string $modelLabel = 'Travel Agent';

    protected static ?string $pluralModelLabel = 'Travel Agents';

    public static function getNavigationGroup(): ?string
    {
        return 'Management';
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                // Basic Information
                \Filament\Forms\Components\TextInput::make('name')
                    ->required()
                    ->maxLength(255)
                    ->label('Company Name'),
                
                \Filament\Forms\Components\TextInput::make('contact_person_name')
                    ->label('Contact Person Name')
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('email')
                    ->email()
                    ->required()
                    ->maxLength(255)
                    ->label('Company Email'),
                
                \Filament\Forms\Components\TextInput::make('phone')
                    ->tel()
                    ->maxLength(255)
                    ->label('Company Phone'),
                
                \Filament\Forms\Components\Textarea::make('contact_info')
                    ->rows(3)
                    ->label('Contact Information'),
                
                // Business Details
                \Filament\Forms\Components\TextInput::make('gst_number')
                    ->label('GST Number')
                    ->maxLength(255),
                
                \Filament\Forms\Components\Textarea::make('address')
                    ->rows(3)
                    ->label('Company Address'),
                
                // File Upload (now available!)
                \Filament\Forms\Components\FileUpload::make('logo_file')
                    ->label('Company Logo')
                    ->directory('travel-agent-logos')
                    ->image()
                    ->maxSize(2048), // 2MB
                
                // Policies and Terms
                \Filament\Forms\Components\Textarea::make('rules_and_regulations')
                    ->rows(3)
                    ->label('Rules and Regulations'),
                
                \Filament\Forms\Components\Textarea::make('terms_and_conditions')
                    ->rows(5)
                    ->label('Terms and Conditions'),
                
                \Filament\Forms\Components\Textarea::make('cancellation_policy')
                    ->rows(5)
                    ->label('Cancellation Policy'),
                
                // JSON Fields - using Repeater (now available!)
                \Filament\Forms\Components\Repeater::make('inclusions')
                    ->schema([
                        \Filament\Forms\Components\TextInput::make('item')
                            ->label('Included Item')
                            ->required(),
                    ])
                    ->label('Package Inclusions')
                    ->defaultItems(0)
                    ->helperText('Add items included in packages'),
                
                \Filament\Forms\Components\Repeater::make('exclusions')
                    ->schema([
                        \Filament\Forms\Components\TextInput::make('item')
                            ->label('Excluded Item')
                            ->required(),
                    ])
                    ->label('Package Exclusions')
                    ->defaultItems(0)
                    ->helperText('Add items excluded from packages'),
                
                // Email Configuration
                \Filament\Forms\Components\TextInput::make('mail_mailer')
                    ->label('Mailer')
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('mail_host')
                    ->label('Mail Host')
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('mail_port')
                    ->label('Mail Port')
                    ->numeric(),
                
                \Filament\Forms\Components\TextInput::make('mail_username')
                    ->label('Mail Username')
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('mail_password')
                    ->label('Mail Password')
                    ->password()
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('mail_from_address')
                    ->label('From Address')
                    ->email()
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('mail_from_name')
                    ->label('From Name')
                    ->maxLength(255),
                
                // SMS Configuration
                \Filament\Forms\Components\TextInput::make('sms_sid')
                    ->label('SMS SID')
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('sms_token')
                    ->label('SMS Token')
                    ->password()
                    ->maxLength(255),
                
                \Filament\Forms\Components\TextInput::make('sms_from_number')
                    ->label('From Number')
                    ->maxLength(255),
                
                // Status
                \Filament\Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->helperText('Whether this travel agent is active'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('contact_person_name')
                    ->label('Contact Person')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable(),
                Tables\Columns\TextColumn::make('phone')
                    ->searchable(),
                Tables\Columns\TextColumn::make('gst_number')
                    ->label('GST No')
                    ->searchable(),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('Status')
                    ->trueLabel('Active')
                    ->falseLabel('Inactive')
                    ->queries(
                        true: fn (Builder $query) => $query->where('is_active', true),
                        false: fn (Builder $query) => $query->where('is_active', false),
                    ),
            ])
            ->actions([
                Actions\EditAction::make(),
                Actions\Action::make('toggleStatus')
                    ->label(fn (TravelAgent $record) => $record->is_active ? 'Deactivate' : 'Activate')
                    ->color(fn (TravelAgent $record) => $record->is_active ? 'danger' : 'success')
                    ->action(function (TravelAgent $record) {
                        $record->update(['is_active' => !$record->is_active]);
                    })
                    ->requiresConfirmation(),
                Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                    Actions\BulkAction::make('activate')
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->requiresConfirmation(),
                    Actions\BulkAction::make('deactivate')
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->requiresConfirmation(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            \App\Filament\Resources\TravelAgents\RelationManagers\UsersRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTravelAgents::route('/'),
            'create' => Pages\CreateTravelAgent::route('/create'),
            'edit' => Pages\EditTravelAgent::route('/{record}/edit'),
        ];
    }
}