<?php

namespace App\Filament\Resources\SubscriptionPlans;

use App\Filament\Resources\SubscriptionPlans\Pages\CreateSubscriptionPlan;
use App\Filament\Resources\SubscriptionPlans\Pages\EditSubscriptionPlan;
use App\Filament\Resources\SubscriptionPlans\Pages\ListSubscriptionPlans;
use App\Models\AiPlan;
use BackedEnum;
use Filament\Actions;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use UnitEnum;

class SubscriptionPlanResource extends Resource
{
    protected static ?string $model = AiPlan::class;

    protected static BackedEnum|string|null $navigationIcon = 'heroicon-o-cube';

    protected static ?string $navigationLabel = 'Subscription Plans';

    protected static ?string $modelLabel = 'Subscription Plan';

    protected static ?string $pluralModelLabel = 'Subscription Plans';

    public static function getNavigationGroup(): ?string
    {
        return 'Billing';
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                // Plan Identification
                \Filament\Forms\Components\TextInput::make('name')
                    ->required()
                    ->maxLength(255)
                    ->label('Plan Name (Internal)')
                    ->helperText('Internal identifier, e.g., "basic", "pro"'),
                
                \Filament\Forms\Components\TextInput::make('display_name')
                    ->required()
                    ->maxLength(255)
                    ->label('Display Name')
                    ->helperText('User-facing plan name'),
                
                \Filament\Forms\Components\TextInput::make('sort_order')
                    ->numeric()
                    ->default(0)
                    ->label('Sort Order'),
                
                // Word Limits and Pricing
                \Filament\Forms\Components\TextInput::make('monthly_words')
                    ->numeric()
                    ->required()
                    ->label('Monthly Words')
                    ->helperText('Number of words included per month'),
                
                \Filament\Forms\Components\TextInput::make('free_words_included')
                    ->numeric()
                    ->default(500)
                    ->label('Free Words Included')
                    ->helperText('Additional free words every month'),
                
                \Filament\Forms\Components\TextInput::make('carry_forward_months')
                    ->numeric()
                    ->default(0)
                    ->label('Carry Forward Months')
                    ->helperText('How many months unused words carry forward'),
                
                \Filament\Forms\Components\TextInput::make('price')
                    ->numeric()
                    ->required()
                    ->prefix('₹')
                    ->step(0.01)
                    ->label('Monthly Price')
                    ->helperText('Monthly subscription price'),
                
                \Filament\Forms\Components\TextInput::make('top_up_rate')
                    ->numeric()
                    ->required()
                    ->prefix('₹')
                    ->step(0.001)
                    ->label('Top-up Rate')
                    ->helperText('Cost per word for additional top-ups'),
                
                // Description and Features
                \Filament\Forms\Components\Textarea::make('description')
                    ->rows(3)
                    ->label('Description')
                    ->helperText('Plan description for customers'),
                
                // Features (JSON field using Repeater)
                \Filament\Forms\Components\Repeater::make('features')
                    ->schema([
                        \Filament\Forms\Components\TextInput::make('feature')
                            ->label('Feature')
                            ->required(),
                    ])
                    ->label('Plan Features')
                    ->defaultItems(0)
                    ->helperText('List of features to display'),
                
                // Status
                \Filament\Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->helperText('Whether this plan is available for subscription'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('display_name')
                    ->searchable()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('monthly_words')
                    ->label('Monthly Words')
                    ->numeric()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('free_words_included')
                    ->label('Free Words')
                    ->numeric()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\TextColumn::make('price')
                    ->label('Price')
                    ->money('INR')
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('top_up_rate')
                    ->label('Top-up Rate')
                    ->money('INR')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\TextColumn::make('carry_forward_months')
                    ->label('Carry Forward')
                    ->numeric()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\TextColumn::make('sort_order')
                    ->label('Order')
                    ->numeric()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('Status')
                    ->trueLabel('Active Only')
                    ->falseLabel('Inactive Only'),
            ])
            ->actions([
                Actions\EditAction::make(),
                Actions\Action::make('toggleStatus')
                    ->label(fn (AiPlan $record) => $record->is_active ? 'Deactivate' : 'Activate')
                    ->color(fn (AiPlan $record) => $record->is_active ? 'danger' : 'success')
                    ->action(function (AiPlan $record) {
                        $record->update(['is_active' => !$record->is_active]);
                    })
                    ->requiresConfirmation(),
                Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                    Actions\BulkAction::make('activate')
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->requiresConfirmation(),
                    Actions\BulkAction::make('deactivate')
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->requiresConfirmation(),
                ]),
            ])
            ->defaultSort('sort_order', 'asc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListSubscriptionPlans::route('/'),
            'create' => CreateSubscriptionPlan::route('/create'),
            'edit' => EditSubscriptionPlan::route('/{record}/edit'),
        ];
    }
}